
// src/ai/flows/dynamic-content-suggestions.ts
'use server';

/**
 * @fileOverview Provides AI-powered suggestions for questions a user can ask on WhatsApp.
 *
 * - suggestWhatsappQuestions - A function that suggests relevant questions.
 * - SuggestWhatsappQuestionsInput - The input type for the suggestWhatsappQuestions function.
 * - SuggestWhatsappQuestionsOutput - The return type for the suggestWhatsappQuestions function.
 */

import {ai} from '@/ai/genkit';
import {z} from 'genkit';

// Input schema remains the same as it provides context for question generation
const SuggestWhatsappQuestionsInputSchema = z.object({
  userBehavior: z
    .string()
    .describe(
      'A description of the user behavior on the website, including pages visited, time spent, and actions taken.'
    ),
  industryTrends: z
    .string()
    .describe('Current industry trends related to financial inclusion and enterprise development.'),
  availableContent: z
    .string()
    .describe(
      'A list of available content (blog posts, service offerings, project details) with brief descriptions, which can be used as context to generate relevant questions.'
    ),
});
export type SuggestWhatsappQuestionsInput = z.infer<typeof SuggestWhatsappQuestionsInputSchema>;

const SuggestedQuestionSchema = z.object({
  question: z.string().describe("A concise and relevant question the user can ask on WhatsApp for more information or engagement related to IntelliWealth Hub's offerings."),
});

const SuggestWhatsappQuestionsOutputSchema = z.object({
  suggestedQuestions: z
    .array(SuggestedQuestionSchema)
    .describe('A list of suggested questions for WhatsApp.'),
});
export type SuggestWhatsappQuestionsOutput = z.infer<typeof SuggestWhatsappQuestionsOutputSchema>;

export async function suggestWhatsappQuestions(input: SuggestWhatsappQuestionsInput): Promise<SuggestWhatsappQuestionsOutput> {
  return suggestWhatsappQuestionsFlow(input);
}

const prompt = ai.definePrompt({
  name: 'suggestWhatsappQuestionsPrompt',
  input: {schema: SuggestWhatsappQuestionsInputSchema},
  output: {schema: SuggestWhatsappQuestionsOutputSchema},
  prompt: `You are an AI assistant for IntelliWealth Hub. Your goal is to help users engage further by suggesting questions they can easily ask on WhatsApp.

Based on the user's website behavior, current industry trends, and IntelliWealth Hub's available content/services, generate 2-3 concise and relevant questions. These questions should encourage the user to seek more information or start a conversation related to IntelliWealth Hub's offerings.

User's behavior on the website:
{{userBehavior}}

Current industry trends:
{{industryTrends}}

Available IntelliWealth Hub content/services (use this as context for question topics):
{{availableContent}}

For each suggestion, provide only the question itself within the 'question' field.
Return the suggestions as an array of objects, under the 'suggestedQuestions' key.

Example of a single item in the 'suggestedQuestions' array:
{
  "question": "What are the first steps to digitize my VSLA group with IntelliWealth Hub?"
}
`,
});

const suggestWhatsappQuestionsFlow = ai.defineFlow(
  {
    name: 'suggestWhatsappQuestionsFlow',
    inputSchema: SuggestWhatsappQuestionsInputSchema,
    outputSchema: SuggestWhatsappQuestionsOutputSchema,
  },
  async input => {
    const {output} = await prompt(input);
    return output!;
  }
);

