
'use client';

import Image from "next/image";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { CheckCircle, LayoutDashboard, Banknote, UsersRound, Smartphone, ShieldCheck, Zap, TrendingUp, Rocket, BarChartHorizontalBig } from "lucide-react";
import { APP_NAME } from "@/constants";
import Link from "next/link";
// import type { Metadata } from 'next'; // Cannot export metadata from client component
import { useState } from "react";
import { RadioGroup, RadioGroupItem } from "@/components/ui/radio-group";
import { Label } from "@/components/ui/label";
import { Badge } from "@/components/ui/badge";

// Metadata needs to be static for client components or moved to a server component parent,
// or a specific layout.tsx for this route.
// export const metadata: Metadata = {
//   title: 'IntelliCash - SaaS for Chama Financial Management',
//   description: `Discover IntelliCash, a cloud-based SaaS platform by ${APP_NAME}, designed to digitize Chama operations with compliant lending, M-Pesa integration, and member management. Explore features and pricing.`,
// };

const dualPurposeFeatures = [
  {
    title: "Regulatory-Compliant Lending System",
    icon: ShieldCheck,
    points: [
      "Operates under Central Bank of Kenya (CBK) digital lending guidelines.",
      "Non-deposit-taking, simplifying regulatory compliance.",
      "Facilitates capped interest rates (e.g., below 18% per annum) as per group policies."
    ]
  },
  {
    title: "Robust Financial Management Tool",
    icon: LayoutDashboard,
    points: [
      "Cloud-based SaaS platform for accessibility and security.",
      "Localized experience tailored for Kenyan Chamas and Saccos.",
      "Seamless M-Pesa integration for instant deposits, withdrawals, and fee payments."
    ]
  }
];

const coreCapabilities = [
  {
    title: "Comprehensive Financial Operations",
    icon: Banknote,
    points: [
      "Savings and share account functionalities.",
      "Diverse loan products with automated interest calculations.",
      "Automated fee management and M-Pesa STK Push transactions."
    ]
  },
  {
    title: "Streamlined Group & Member Management",
    icon: UsersRound,
    points: [
      "KYC-compliant member onboarding processes.",
      "Self-service portals for members to view statements and apply for loans.",
      "Internal communication tools to foster collaboration and transparency."
    ]
  }
];

const whyIntelliCashBenefits = [
  { title: "Enhanced Transparency", description: "Digital records and automated processes reduce errors and improve trust among members.", icon: CheckCircle },
  { title: "Improved Efficiency", description: "Automate contributions, loan processing, and reporting, saving time for Chama officials.", icon: Rocket },
  { title: "Increased Accessibility", description: "Members can access their information and services anytime, anywhere via web and mobile.", icon: Smartphone },
  { title: "Better Financial Control", description: "Gain clear insights into the Chama's financial health with comprehensive dashboards and reports.", icon: BarChartHorizontalBig },
  { title: "Regulatory Adherence", description: "Designed to help Chamas operate within Kenyan financial guidelines for digital lenders.", icon: ShieldCheck },
  { title: "Scalability", description: "Whether you're a small Chama or a growing Sacco, IntelliCash scales with your needs.", icon: TrendingUp },
];

const pricingTiersData = [
  {
    name: "Starter Group",
    monthlyPrice: "Ksh 2,500",
    annualPrice: "Ksh 10,000",
    periodMonthly: "/month",
    periodAnnually: "/year",
    annualSavingsPercentage: 67, // (2500*12 = 30000. 10000/30000 = 0.33. 1-0.33 = 0.67)
    features: [
      "Up to 50 Members",
      "Core Financial Tracking",
      "Basic Loan Management",
      "M-Pesa Integration (Paybill)",
      "Community Support"
    ],
    cta: "Get Started",
    idealFor: "Small, newly formed Chamas."
  },
  {
    name: "Growing Chama",
    monthlyPrice: "Ksh 5,000",
    annualPrice: "Ksh 50,000",
    periodMonthly: "/month",
    periodAnnually: "/year",
    annualSavingsPercentage: 17, // (5000*12 = 60000. 50000/60000 = 0.83. 1-0.83 = 0.17)
    features: [
      "Up to 200 Members",
      "Advanced Loan Products",
      "Member Self-Service Portal",
      "Automated Reporting",
      "Priority Email Support"
    ],
    cta: "Choose Plan",
    popular: true,
    idealFor: "Established Chamas looking to scale."
  },
  {
    name: "Large Sacco / Group",
    customPrice: "Custom",
    features: [
      "Unlimited Members",
      "Custom Feature Development",
      "Dedicated Account Manager",
      "Advanced Security Options",
      "On-site Training"
    ],
    cta: "Contact Sales",
    idealFor: "Large Saccos or groups with specific needs."
  }
];

export default function IntelliCashPage() {
  const [billingCycle, setBillingCycle] = useState<'monthly' | 'annually'>('monthly');

  return (
    <div className="bg-background text-foreground">
      {/* Hero Section */}
      <section className="py-20 md:py-28 bg-gradient-to-b from-primary/5 via-primary/10 to-background">
        <div className="container mx-auto px-4 text-center">
          <div className="mb-6 inline-flex items-center justify-center p-3 rounded-lg shadow-md dark:bg-white dark:shadow-slate-700/50">
            <Image
              src="https://files.tari.africa/api.php?path=public&name=1748418365_6836bf3d803fc.png"
              alt="IntelliCash Logo"
              width={180} 
              height={60} 
              className="object-contain"
              data-ai-hint="fintech product logo"
            />
          </div>
          <h1 className="text-4xl md:text-5xl font-bold text-primary mb-4 text-shadow-primary">IntelliCash</h1>
          <p className="text-lg md:text-xl text-muted-foreground max-w-3xl mx-auto mb-2">
            by {APP_NAME}
          </p>
          <p className="text-xl md:text-2xl text-foreground max-w-3xl mx-auto">
            The Smart SaaS Platform for Chama Financial Management in Kenya.
          </p>
        </div>
      </section>

      {/* Overview Section */}
      <section className="py-16 lg:py-24">
        <div className="container mx-auto px-4">
          <div className="grid md:grid-cols-2 gap-12 items-center">
            <div className="order-2 md:order-1">
              <h2 className="text-3xl font-bold text-foreground mb-6">What is IntelliCash?</h2>
              <p className="text-muted-foreground mb-4">
                IntelliCash is a cutting-edge, cloud-based Software-as-a-Service (SaaS) platform meticulously designed by {APP_NAME}. It's tailored specifically for Chamas—informal community savings groups—and growing Saccos that are a cornerstone of financial activity in Kenya.
              </p>
              <p className="text-muted-foreground mb-4">
                Our mission with IntelliCash is to revolutionize how these groups operate by digitizing their processes. This SaaS solution enhances transparency, expands financial access for members, and boosts overall efficiency through modern fintech solutions, accessible anytime, anywhere.
              </p>
              <p className="text-muted-foreground">
                By leveraging technology, IntelliCash empowers Chamas and Saccos to manage their finances more effectively, make informed decisions, and provide greater value to their members with a secure and scalable system.
              </p>
            </div>
            <div className="order-1 md:order-2">
              <Image
                src="https://files.tari.africa/api.php?path=public&name=1748417881_6836bd59443b5.png"
                alt="IntelliCash Platform Interface - SaaS for Chamas"
                width={600}
                height={450}
                className="rounded-lg shadow-xl object-cover"
                data-ai-hint="fintech platform user"
              />
            </div>
          </div>
        </div>
      </section>

      {/* Key Features & Benefits Section */}
      <section className="py-16 lg:py-24 bg-muted/30">
        <div className="container mx-auto px-4">
          <div className="text-center mb-12">
            <h2 className="text-3xl lg:text-4xl font-bold text-foreground">Powerful Features for Your Group</h2>
            <p className="mt-4 text-lg text-muted-foreground max-w-2xl mx-auto">
              IntelliCash is packed with robust features designed to streamline group operations and enhance member experience.
            </p>
          </div>

          <div className="mb-16">
            <h3 className="text-2xl font-semibold text-primary text-center mb-8 text-shadow-primary">Core Platform Pillars</h3>
            <div className="grid md:grid-cols-1 lg:grid-cols-2 gap-8">
              {dualPurposeFeatures.map((feature) => (
                <Card key={feature.title} className="shadow-lg h-full flex flex-col">
                  <CardHeader className="flex-row items-center gap-4">
                    <feature.icon className="h-10 w-10 text-accent flex-shrink-0" />
                    <CardTitle className="text-xl text-primary text-shadow-primary">{feature.title}</CardTitle>
                  </CardHeader>
                  <CardContent className="flex-grow">
                    <ul className="space-y-2">
                      {feature.points.map((point, index) => (
                        <li key={index} className="flex items-start">
                          <CheckCircle className="h-5 w-5 text-green-500 mr-2 mt-0.5 flex-shrink-0" />
                          <span className="text-sm text-muted-foreground">{point}</span>
                        </li>
                      ))}
                    </ul>
                  </CardContent>
                </Card>
              ))}
            </div>
          </div>

          <div>
            <h3 className="text-2xl font-semibold text-primary text-center mb-8 text-shadow-primary">Comprehensive Capabilities</h3>
            <div className="grid md:grid-cols-1 lg:grid-cols-2 gap-8">
              {coreCapabilities.map((capability) => (
                <Card key={capability.title} className="shadow-lg h-full flex flex-col">
                  <CardHeader className="flex-row items-center gap-4">
                    <capability.icon className="h-10 w-10 text-accent flex-shrink-0" />
                    <CardTitle className="text-xl text-primary text-shadow-primary">{capability.title}</CardTitle>
                  </CardHeader>
                  <CardContent className="flex-grow">
                    <ul className="space-y-2">
                      {capability.points.map((point, index) => (
                        <li key={index} className="flex items-start">
                          <CheckCircle className="h-5 w-5 text-green-500 mr-2 mt-0.5 flex-shrink-0" />
                           <span className="text-sm text-muted-foreground">{point}</span>
                        </li>
                      ))}
                    </ul>
                  </CardContent>
                </Card>
              ))}
            </div>
          </div>
        </div>
      </section>

      {/* Why IntelliCash Section */}
      <section className="py-16 lg:py-24 bg-background">
        <div className="container mx-auto px-4">
          <div className="text-center mb-12">
            <h2 className="text-3xl lg:text-4xl font-bold text-foreground">Why Choose IntelliCash?</h2>
            <p className="mt-4 text-lg text-muted-foreground max-w-2xl mx-auto">
              Empower your Chama or Sacco with a modern, reliable, and user-friendly SaaS solution.
            </p>
          </div>
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
            {whyIntelliCashBenefits.map((benefit) => (
              <Card key={benefit.title} className="shadow-md hover:shadow-lg transition-shadow duration-300 bg-card">
                <CardHeader className="items-center text-center">
                  <benefit.icon className="h-10 w-10 text-primary mb-3" />
                  <CardTitle className="text-xl text-primary text-shadow-primary">{benefit.title}</CardTitle>
                </CardHeader>
                <CardContent className="text-center">
                  <p className="text-sm text-muted-foreground">{benefit.description}</p>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </section>
      
      {/* Pricing Section */}
      <section className="py-16 lg:py-24 bg-primary/5">
        <div className="container mx-auto px-4">
          <div className="text-center mb-12">
            <h2 className="text-3xl lg:text-4xl font-bold text-foreground">Flexible Pricing for Every Group</h2>
            <p className="mt-4 text-lg text-muted-foreground max-w-2xl mx-auto">
              Choose the plan that best fits your Chama's or Sacco's size and needs.
            </p>
          </div>

          <div className="flex justify-center mb-10">
            <RadioGroup
              value={billingCycle}
              onValueChange={(value: string) => setBillingCycle(value as 'monthly' | 'annually')}
              className="grid grid-cols-2 gap-1 rounded-lg bg-muted p-1 w-auto shadow"
            >
              <div>
                <RadioGroupItem value="monthly" id="monthly" className="sr-only" />
                <Label
                  htmlFor="monthly"
                  className={`cursor-pointer rounded-md px-6 py-2.5 text-sm font-medium transition-colors block text-center
                    ${billingCycle === 'monthly' ? 'bg-primary text-primary-foreground shadow-sm' : 'hover:bg-background/70 text-muted-foreground'}`}
                >
                  Monthly
                </Label>
              </div>
              <div>
                <RadioGroupItem value="annually" id="annually" className="sr-only" />
                <Label
                  htmlFor="annually"
                  className={`cursor-pointer rounded-md px-6 py-2.5 text-sm font-medium transition-colors block text-center
                    ${billingCycle === 'annually' ? 'bg-primary text-primary-foreground shadow-sm' : 'hover:bg-background/70 text-muted-foreground'}`}
                >
                  Annually
                  <Badge variant="outline" className="ml-2 border-accent text-accent bg-accent/10 text-xs px-1.5 py-0.5">Save!</Badge>
                </Label>
              </div>
            </RadioGroup>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8 items-stretch">
            {pricingTiersData.map((tier) => (
              <Card key={tier.name} className={`flex flex-col shadow-xl ${tier.popular ? 'border-2 border-primary ring-2 ring-primary/50' : 'border-border'}`}>
                {tier.popular && (
                  <div className="py-1 px-4 bg-primary text-primary-foreground text-sm font-semibold text-center rounded-t-md">
                    Most Popular
                  </div>
                )}
                <CardHeader className="text-center">
                  <CardTitle className="text-2xl text-primary text-shadow-primary">{tier.name}</CardTitle>
                  <p className="text-sm text-muted-foreground">{tier.idealFor}</p>
                  <div className="my-4">
                    <span className="text-4xl font-bold text-foreground">
                      {tier.customPrice 
                        ? tier.customPrice 
                        : (billingCycle === 'monthly' ? tier.monthlyPrice : tier.annualPrice)}
                    </span>
                    {tier.customPrice ? null : (
                      <span className="text-muted-foreground">
                        {billingCycle === 'monthly' ? tier.periodMonthly : tier.periodAnnually}
                      </span>
                    )}
                    {billingCycle === 'annually' && tier.annualSavingsPercentage && !tier.customPrice && (
                       <p className="text-xs text-accent font-medium mt-1">(Save ~{tier.annualSavingsPercentage}%)</p>
                    )}
                  </div>
                </CardHeader>
                <CardContent className="flex-grow">
                  <ul className="space-y-3">
                    {tier.features.map((feature, index) => (
                      <li key={index} className="flex items-center">
                        <CheckCircle className="h-5 w-5 text-green-500 mr-2 flex-shrink-0" />
                        <span className="text-sm text-muted-foreground">{feature}</span>
                      </li>
                    ))}
                  </ul>
                </CardContent>
                <CardDescription className="p-6 pt-0 mt-auto">
                  <Button 
                    asChild 
                    className={`w-full text-lg py-3 ${tier.popular ? '' : 'bg-accent text-accent-foreground hover:bg-accent/90'}`}
                    variant={tier.popular ? 'default' : 'secondary'}
                  >
                    <Link href={`/contact?subject=IntelliCash+${tier.name}+Plan+(${billingCycle})`}>{tier.cta}</Link>
                  </Button>
                </CardDescription>
              </Card>
            ))}
          </div>
           <p className="text-center text-muted-foreground mt-8">
            Need a custom solution or have more questions? <Link href="/contact?subject=IntelliCash+Pricing+Inquiry" className="text-primary hover:underline">Contact our sales team</Link>.
          </p>
        </div>
      </section>

      {/* M-Pesa Integration Highlight */}
      <section className="py-16 lg:py-24">
        <div className="container mx-auto px-4">
          <div className="text-center bg-primary/10 p-8 md:p-12 rounded-xl shadow-lg">
              <Smartphone className="h-16 w-16 text-primary mx-auto mb-6" />
              <h2 className="text-3xl font-bold text-primary mb-4 text-shadow-primary">Seamless M-Pesa Integration</h2>
              <p className="text-lg text-muted-foreground max-w-2xl mx-auto mb-6">
                IntelliCash offers deep integration with M-Pesa, Kenya's leading mobile money service. This enables instant deposits, withdrawals, and fee payments directly within the platform, providing unparalleled convenience for Chama members.
              </p>
              <Button size="lg" asChild>
                <Link href="/contact?subject=IntelliCash+M-Pesa+Integration">Discover More</Link>
              </Button>
          </div>
        </div>
      </section>

      {/* Call to Action Section */}
      <section className="py-16 lg:py-24 bg-primary text-primary-foreground">
        <div className="container mx-auto px-4 text-center">
          <h2 className="text-3xl font-bold mb-6">Ready to Digitize Your Chama with Our SaaS Platform?</h2>
          <p className="text-lg text-primary-foreground/80 mb-8 max-w-2xl mx-auto">
            Take your Chama to the next level with IntelliCash. Enhance transparency, improve efficiency, and unlock new financial opportunities for your members.
          </p>
          <Button variant="secondary" size="lg" asChild className="bg-accent text-accent-foreground hover:bg-accent/90 text-lg px-8 py-3">
            <Link href="/contact?subject=IntelliCash+Demo+Request">Request a Demo</Link>
          </Button>
        </div>
      </section>
    </div>
  );
}

// For client components that need metadata, it should ideally be exported from a parent server component or page.tsx
// If this page *must* be a client component at the root, metadata can be set dynamically using `useEffect` and document.title,
// but this is not the recommended Next.js 13+ App Router approach for SEO.
// For now, I've commented out the static metadata export.
// You can create a `src/app/intellicash/layout.tsx` or ensure this page is wrapped by a server component setting metadata.
// export const metadata: Metadata = { // Removed export
//   title: 'IntelliCash - SaaS for Chama Financial Management',
//   description: `Discover IntelliCash, a cloud-based SaaS platform by ${APP_NAME}, designed to digitize Chama operations with compliant lending, M-Pesa integration, and member management. Explore features and pricing.`,
// };


    
